using System;

class InversionCountAVLTree
{
    public class Node
    {
        public float key;
        public Node left, right;
        public int size, height;

        public Node(float val)
        {
            key = val;
            left = null;
            right = null;
            height = 1;
            size = 1;
        }
    }

    public int Result { get; private set; }
    private Node root;

    public InversionCountAVLTree()
    {
        root = null;
    }

    static int Height(Node N)
    {
        if (N == null)
            return 0;
        return N.height;
    }

    static int Size(Node N)
    {
        if (N == null)
            return 0;
        return N.size;
    }

    static Node RightRotate(Node y)
    {
        var x = y.left;
        var T2 = x.right;

        x.right = y;
        y.left = T2;

        y.height = Math.Max(Height(y.left), Height(y.right)) + 1;
        x.height = Math.Max(Height(x.left), Height(x.right)) + 1;

        y.size = Size(y.left) + Size(y.right) + 1;
        x.size = Size(x.left) + Size(x.right) + 1;

        return x;
    }

    static Node LeftRotate(Node x)
    {
        var y = x.right;
        var T2 = y.left;

        y.left = x;
        x.right = T2;

        x.height = Math.Max(Height(x.left), Height(x.right)) + 1;
        y.height = Math.Max(Height(y.left), Height(y.right)) + 1;

        x.size = Size(x.left) + Size(x.right) + 1;
        y.size = Size(y.left) + Size(y.right) + 1;

        return y;
    }

    static int GetBalance(Node N)
    {
        if (N == null)
            return 0;
        return Height(N.left) - Height(N.right);
    }

    public void Insert(float key)
    {
        root = Insert(root, key);
    }

    private Node Insert(Node node, float key)
    {
        if (node == null)
            return new Node(key);
        if (key <= node.key)
        {
            node.left = Insert(node.left, key);
            Result += Size(node.right) + 1;
        }
        else
        {
            node.right = Insert(node.right, key);
        }

        node.height = Math.Max(Height(node.left), Height(node.right)) + 1;
        node.size = Size(node.left) + Size(node.right) + 1;

        int balance = GetBalance(node);

        if (balance > 1 && key < node.left.key)
        {
            return RightRotate(node);
        }

        if (balance < -1 && key > node.right.key)
        {
            return LeftRotate(node);
        }

        if (balance > 1 && key > node.left.key)
        {
            node.left = LeftRotate(node.left);
            return RightRotate(node);
        }

        if (balance < -1 && key < node.right.key)
        {
            node.right = RightRotate(node.right);
            return LeftRotate(node);
        }

        return node;
    }

    public void Clear()
    {
        root = ClearRecursive(root);
        Result = 0;
    }

    private static Node ClearRecursive(Node node)
    {
        if (node == null)
            return null;

        node.left = ClearRecursive(node.left);
        node.right = ClearRecursive(node.right);

        return null;
    }
}

class Program
{
    class Trkac
    {
        public float Brzina { get; private set; }
        public float Pozicija { get; private set; }

        public Trkac(float pozicija, float brzina)
        {
            Brzina = brzina;
            Pozicija = pozicija;
        }

        public float VremeDoPozicije(float pozicija)
        {
            return (pozicija - Pozicija) / Brzina;
        }

        public float NovaPozicija(float vreme)
        {
            return Pozicija + vreme * Brzina;
        }
    }

    const float preciznost = 0.000001f;
    static InversionCountAVLTree drvo = new InversionCountAVLTree();

    static int BrojPreticanjaNaPoziciji(int brojTrkaca, Trkac[] trkaci, float pozicija)
    {
        var vreme = trkaci[0].VremeDoPozicije(pozicija);

        for(int i = 0; i < brojTrkaca ; i++)
        {
            var zaDodavanje = trkaci[i].NovaPozicija(vreme);
            drvo.Insert(zaDodavanje);
        }
        var preticanja = drvo.Result;
        drvo.Clear();
        return preticanja;
    }

    static float VremeDoPreticanja(int brojTrkaca, Trkac[] trkaci, int brojPreticanja)
    {
        var levi = 0f;
        var desni = float.MaxValue;
        var resenje = float.MaxValue;
        var nulti = trkaci[0];

        while ((desni - levi) > preciznost)
        {
            var srednji = (desni - levi) / 2 + levi;
            var preticanja = BrojPreticanjaNaPoziciji(brojTrkaca, trkaci, srednji);

            if (preticanja < brojPreticanja)
            {
                levi = srednji + preciznost;
            }
            else
            {
                desni = srednji - preciznost;
            }

            resenje = nulti.VremeDoPozicije(srednji);
        }

        return resenje;
    }

    static void Main()
    {
        var dimenzija = Console.ReadLine().Split(' ');
        var brojTrkaca = int.Parse(dimenzija[0]);
        var brojPreticanja = int.Parse(dimenzija[1]);

        var trkaci = new Trkac[brojTrkaca];
        for(int i = 0; i < brojTrkaca; i++)
        {
            var podaci = Console.ReadLine().Split(' ');
            trkaci[i] = new Trkac(float.Parse(podaci[0]), float.Parse(podaci[1]));
        }

        var resenje = VremeDoPreticanja(brojTrkaca, trkaci, brojPreticanja);
        Console.WriteLine(resenje.ToString(".000000000"));
    }
}